classdef Frame
    %classdef Frame - implements class for cartesian frame transformations
    % for use with classes vct3 and RotMx.
    %
    % Copyright (C) Russell H. Taylor, Johns Hopkins University, 2013
    %
    %
    % Properties
    %   R                   RotMx - the rotational component
    %   p                   vct3  - the translational component
    %
    % Methods
    %
    % Notation
    %   F, F1, ...          Frame objects
    %   R, R1, R2, ...      RotMx objects
    %   p, v, v1, v2, ...   vct3 objects
    %   eX, eY, eZ          unit X, Y, Z vectors
    %
    % Constructors
    %
    % F = Frame(R,p)        Construct frame from components
    % F = Frame(a,b,c)      Construct frame from 3 vct3 points
    % F = Frame(F1)         Copy constructor
    % F = Frame.eye()       Indentity
    %
    % Computation
    %
    % v = F1*v2             Computes F1*v2 = R1*v2+p1
    % F = F1*F2             Computes F1*F2 = Frame(R1*R2,R1*p2+p1)
    %
    % F = F1.Inverse()      Computes Frame(R1.Inverse(),-(R1.Inverse()*p1))
    % v = mtimes(F1,v1)     Computes F1*v2 = R1*v2+p1
    % F = mtimes(F1,F2)     Computes F1*F2 = Frame(R1*R2,R1*p2+p1)
    %
    % Other methods
    %   display(F)          Console display (calls disp)
    %   disp(F)             Displays frame as a three row matrix, with the
    %                       first 3 columns showing R and rest showing p
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor
    
    properties
        R = RotMx();
        p = vct3();
    end
    
    methods
        function F = Frame(varargin)
            switch nargin
                case 0
                    % just make one
                    return;
                case 1
                    if isa(varargin{1},'Frame')
                        F.R=varargin{1}.R;
                        F.p=varargin{1}.p;
                        return;
                    elseif isa(varargin{1},'vct3')
                        R = RotMx(vargin{1}.el');
                        return;
                    else
                        error('BadArg','Bad Argument to Frame',varargin{1});                      
                            % this should be an error case for now
                    end
                case 2
                   if isa(varargin{1},'RotMx') && isa(varargin{2},'vct3')
                       F.R=varargin{1};
                       F.p=varargin{2};
                       return;
                   else
                       %%% could add more cases to allow to pass elements in
                       %%% This would let us make internal calculations more
                       %%% efficient (e.g., F1*F2) by eliminating some
                       %%% copying
                       error('BadArg','Bad Arguments to Frame',varargin);
                   end
                otherwise
                    error('BadArg','Bad Arguments to Frame',varargin);
            end
        end
        
        function Fv = mtimes(F,v)
            switch class(v)
                case 'vct3'
                    Fv = vct3(bsxfun(@plus,F.R.el*v.el,F.p.el));
                case 'vct3Array'
                    Fv = vct3Array(bsxfun(@plus,F.R.el*v.el,F.p.el));
                case 'Frame'
                    Fv = Frame(F.R*v.R,F*v.p);
                otherwise  
                    error('BadArg','Bad Argument to Frame multiply',v);
            end
        end
        
        function Fi = Inverse(F)
            Fi = Frame(RotMx(F.R.el'),vct3(-(F.R.el'*F.p.el)));
        end
                              
        function disp(X)
             % disp(horzcat(repmat('[',3,1),X.R.el,repmat('|',3,1),X.p.el,repmat(']',3,1)));
             for i=1:size(X,1)
                disp(horzcat(X(i).R.el,X(i).p.el));
             end
        end
        
        function display(X)          
            if isequal(get(0,'FormatSpacing'),'compact')
               disp([inputname(1) ' =']);
               disp(X)
            else
               disp(' ')
               disp([inputname(1) ' =']);
               disp(' ');
               disp(X);
            end
        end
    end
    
    methods (Static)
        function F = eye()
            F = Frame(RotMx.eye(),vct3.null());
        end
    end
    
end

